%% Example 6: Pulse Streamer Continuous Streaming
% This script demonstrates the continuous streaming feature of the
% Swabian Pulse Streamer 8/2 using the 'upload' and 'start' methods.
%
% Overview:
% This example shows how to use the continuous streaming mode of the Pulse Streamer 8/2.
% With firmware release 2.x, the Pulse Streamer 8/2 introduces a new feature for continuous
% streaming, allowing data to be streamed seamlessly without interruptions.
% The current example uses the preliminary release 2.0.0 Beta2, which offers an opportunity
% to test the continuous streaming feature while the final firmware release is being completed.
% The API is fully backward compatible, and the official release will not change the API.
%
% How It Works:
% The Pulse Streamer 8/2 is equipped with two independent memory slots that enable continuous
% streaming of pulse sequences. While data is being streamed from one memory slot, the next set
% of data can be uploaded to the other slot, ensuring uninterrupted output. The transition process
% between memory slots is flexible and can be configured to either switch seamlessly to the new data,
% hold in an idle state, or repeat the current data until new data becomes available.
%
% This script demonstrates how to create a simple pulse sequence, upload it to the Pulse Streamer,
% and continuously stream the sequence while waiting for new data after each slot.
%
% Documentation:
% For a detailed description of the Pulse Streamer API and its features, please refer to:
% https://www.swabianinstruments.com/static/documentation/pulse-streamer/v2.0/
%
% Requirements:
% - Ensure that the Pulse Streamer 8/2 is properly connected to your network.
% - Install the PulseStreamer client package from Swabian Instruments' website or MathWorks Exchange.
% - Firmware version 2.0.0 Beta2 or later is required for continuous streaming.

%% Connect to Pulse Streamer

% Set the IP address of the Pulse Streamer. Update this line as needed.
ip_hostname = 'pulsestreamer';  % Replace with the correct IP address

HIGH = 1;
LOW = 0;

% Attempt to connect to the Pulse Streamer using the provided IP address
try
    pulser = PulseStreamer.PulseStreamer(ip_hostname);
catch
    % Provide more detailed error messages in case of failure
    fprintf('Unable to connect to Pulse Streamer at IP address: %s\n', ip_hostname);
    fprintf('Please check the IP address and network connection.\n');
    fprintf('\nAvailable Pulse Streamer devices found on the network:\n');
    disp(PulseStreamer.findPulseStreamers());
    fprintf('Please select one of the devices listed above or visit https://www.swabianinstruments.com for assistance.\n');
    
    % Terminate execution
    error('No Pulse Streamer found. Program will terminate.');
end

%% Streaming continuously a continuous pattern on ch_0 and a signal with a shifting period on ch_1
% This example demonstrates how a long sequence that does not fit into
% PulseStreamer memory can be streamed seamlessly in suitable chunks.
% For demonstration purpose the generated signals are constant period
% pulses on one channel and ramp-up ramp-down pulse period sweep on another
% channel.

disp('******************************************');
disp('****** Streaming data continuously ******');
disp('******************************************');
disp(' ');
disp('Stream data continuously and stop afterwards');
disp(' ');

ref_period = 10000; % 100 kHz
min_shift_period = 30000; % Minimum pulse period in nanoseconds
max_shift_period = 90000; % Maximum pulse period in nanoseconds
gate_width = 100; % pulse duration (constant)
total_duration = 10e9; % total duration of the pulse sequence

ref_ch = 0;
shift_ch = 1;

% Generate patterns for constant frequency and shifting square waves
num_periods = floor(total_duration / ref_period);
reference_pattern = repmat({10, HIGH; ref_period - 10, LOW}, num_periods, 1);

% Preallocate shifting_pattern for better performance
estimated_periods = ceil(total_duration / min_shift_period)*2; % Rough estimate, multiplied by 2 for HIGH and LOW entries
shifting_pattern = cell(estimated_periods, 2); % Preallocate with an estimated size

% Precompute constants for ramp-up ramp-down period
cycle_period_ns = total_duration; 
half_cycle = cycle_period_ns / 2;
range = max_shift_period - min_shift_period;
slope = range / half_cycle;

cumulative_time = 0;
index = 1;

% Calculate complete sequence. 
while cumulative_time <= total_duration
    % Calculate the current period value using the triangular wave function
    % phase of the cycle in the range [0, cycle_period_ns)
    phase_ns = mod(cumulative_time, cycle_period_ns);

    % Determine ramp value based on phase position
    if phase_ns < half_cycle
        % Ramp-up: Linearly increase from min_val to max_val
        period = min_shift_period + slope * phase_ns;
    else
        % Ramp-down: Linearly decrease from max_val to min_val
        period = max_shift_period - slope * (phase_ns - half_cycle);
    end
    
    % Update the shifting_pattern (using the preallocated array)
    shifting_pattern{index, 1} = gate_width;
    shifting_pattern{index, 2} = HIGH;
    index = index + 1;
    
    shifting_pattern{index, 1} = round(period) - gate_width;
    shifting_pattern{index, 2} = LOW;
    index = index + 1;
    
    % Update the cumulative time
    cumulative_time = cumulative_time + period;
end

% Trim unused preallocated cells if necessary
shifting_pattern = shifting_pattern(1:index-1, :);

% Set total sequence, which has many more sequence steps than the Pulse Streamer can accept at once
seq_total = pulser.createSequence();
seq_total.setDigital(ref_ch, reference_pattern);
seq_total.setDigital(shift_ch, shifting_pattern);

% Split the total sequence
seq_chunks = seq_total.split(1e9:1e9:total_duration);

pulser.reset();
input('Press <ENTER> to start streaming', 's');

for i = 1:length(seq_chunks)
    seq_chunk = seq_chunks(i);  % Get the current sequence chunk
    % Alternatively you can avoid creating large sequences and calculate only 
    % necessary sequence chunk here.

    if i == 1
        % Upload first sequence chunk
        pulser.upload(pulser.AUTO, seq_chunk, ...
            'n_runs', 1, ...
            'next_action', PulseStreamer.NextAction.SWITCH_SLOT_EXPECT_NEW_DATA, ...
            'when', PulseStreamer.When.IMMEDIATE);
        
        fprintf('Sequence chunk %d uploaded.\n', i);
        % start streaming
        pulser.start(pulser.AUTO, pulser.REPEAT_INFINITELY);
        disp('Started streaming continuosly...');

    elseif i < length(seq_chunks)
        % Upload next sequence chunk when slot becomes available
        while ~pulser.isReadyForData(pulser.AUTO)
            pause(0.05);
        end
        pulser.upload(pulser.AUTO, seq_chunk, ...
            'n_runs', 1, ...
            'next_action', PulseStreamer.NextAction.SWITCH_SLOT_EXPECT_NEW_DATA, ...
            'when', PulseStreamer.When.IMMEDIATE);
        fprintf('Sequence chunk %d uploaded.\n', i);
    else
        % Upload last sequence chunk
        while ~pulser.isReadyForData(pulser.AUTO)
            pause(0.05);
        end
        pulser.upload(pulser.AUTO, seq_chunk, ...
            'n_runs', 1, ...
            'next_action', PulseStreamer.NextAction.STOP);
        
        disp('Last sequence chunk uploaded.');
        
    end
end

% Wait until streaming is finished
while ~pulser.hasFinished()
    pause(0.5);
end
fprintf('Finished streaming.\n');

input('\nFor next example press <ENTER>', 's');


%% Streaming Three Data Slots and Repeating the Last Memory Slot
disp('**********************************************************************');
disp('****** Streaming three data slots with data block as nested loop ******');
disp('**********************************************************************');
disp(' ');
disp('Streaming three data slots with patterns for setup, measurement, and sync-pulses, and repeating the last data slot afterwards.');
disp(' ');

% Parameters for sequences
period = 80;
laser_pulse_width = 15;
reference_delay = 10;
reference_signal_width = 20;

% Channel assignments
laser_ch = 0;
reference_ch = 1;
mw_ch = 2;
final_slot_trigger_ch = 3;

% Microwave pulse patterns (for the third channel)
delay_sweep = 40:10:70;
mw_pattern = cell(3*numel(delay_sweep),2);
i = 1;
for mw_pulse_width = delay_sweep

    mw_pattern{i, 1} = (period - mw_pulse_width) / 2;
    mw_pattern{i, 2} = LOW;
    i = i+1;

    mw_pattern{i, 1} = mw_pulse_width;
    mw_pattern{i, 2} = HIGH;
    i = i+1;

    mw_pattern{i, 1} = (period - mw_pulse_width) / 2;
    mw_pattern{i, 2} = LOW;
    i = i+1;
end

% Sequence 0 - Initialization Sequence
seq0 = pulser.createSequence();
% Laser pulses on channel 0
seq0.setDigital(laser_ch, {laser_pulse_width, HIGH; period - laser_pulse_width, LOW});  
% Reference signal on channel 1
seq0.setDigital(reference_ch, {reference_delay, LOW; reference_signal_width, HIGH; period - reference_signal_width - reference_delay, LOW});

% Sequence 1 - Measurement Sequence
seq1 = pulser.createSequence();
% Laser pulses on channel 0 (repeated 8 times)
seq1.setDigital(laser_ch, repmat({laser_pulse_width, HIGH; period - laser_pulse_width, LOW}, 8, 1));
% Reference signal on channel 1 (repeated 8 times)
seq1.setDigital(reference_ch, repmat({reference_delay, LOW; reference_signal_width, HIGH; period - reference_signal_width - reference_delay, LOW}, 8, 1));
% Microwave pattern on channel 2
seq1.setDigital(mw_ch, mw_pattern);

% Sequence 2 - Sync-Pulse Sequence (final slot)
seq2 = pulser.createSequence();
% Reference signal on channel 1
seq2.setDigital(reference_ch, {reference_delay, LOW; reference_signal_width, HIGH; period - reference_signal_width - reference_delay, LOW});
% Trigger signal on channel 3 (final slot)
seq2.setDigital(final_slot_trigger_ch, {8, HIGH; 8, LOW});

% Reset Pulse Streamer
pulser.reset();
% Configure trigger to Software. Send trigger by calling "pulser.startNow()"
pulser.setTrigger(PulseStreamer.TriggerStart.SOFTWARE);

% Upload sequences to memory slots
disp('Uploading initial sequences...');
% Upload first sequence with positional parameters
pulser.upload(pulser.AUTO, seq0, ...
    'n_runs', 1, ...
    'next_action', PulseStreamer.NextAction.SWITCH_SLOT_EXPECT_NEW_DATA);

% Upload second sequence, using positional and name-value pairs
pulser.upload(pulser.AUTO, seq1, ...
    'n_runs', 1e6, ...
    'next_action', PulseStreamer.NextAction.SWITCH_SLOT_EXPECT_NEW_DATA);

% Start streaming with 10 slots, playing the last slot once, and then repeating
disp('Starting streaming...');
pulser.start(pulser.AUTO, 10);  

input('Press <ENTER> to start streaming now', 's');
% Send software trigger
pulser.startNow();

fprintf('Pulse Streamer is streaming: %d \n', pulser.isStreaming());

% Wait until the next memory slot is writable
while ~pulser.isReadyForData(pulser.AUTO)
    pause(0.001);
end

% Upload further data to memory (final slot with repeating)
disp('Uploading new data...');
% Upload final sequence using name-value pairs only
pulser.upload(pulser.AUTO, seq2, ...
    'n_runs', 1, ...
    'next_action', PulseStreamer.NextAction.REPEAT_SLOT);

% Wait until the streaming is complete
while pulser.isStreaming()
    pause(0.1);
end

fprintf('Pulse Streamer has finished all three sequence slots: %d \n', pulser.hasFinished());

